﻿using System;
using System.Collections.Generic;
using System.Data;
using System.Linq;
using System.Text;

using Framework.Data;

namespace Framework.Web
{
    /// <summary>
    /// ローカライズのためのテキストを管理/取得するためのクラスです。
    /// リストのデータソースはXMLファイル、またはデータベースなどが利用できます。
    /// XMLファイルをデータソースにする場合は、LTextMgr.LTextSrcXMLFilePath　にパスを指定、その他のデータソースを使用する場合は、
    /// リスナを介してアプリケーション側へリクエストするので、アプリケーション起動時などでリスナの登録が必要です。
    /// LTextMgr.SetListener(リスナのインスタンス)。
    /// </summary>
    public static class LTextMgr
    {
        private const string CACHE_KEY = "Framework.Web.LTextMgr.LTextSrc";

        /// <summary>
        /// データソースにXMLファイルを使用する場合のファイルパス
        /// </summary>
        public static string LTextSrcXMLFilePath { get; set; }

        private static ILTextSrcRequestListener _listener;

        private static LTextSrcDataSet.LTextSrcDataTable LTextSrc
        {
            get
            {
                var src = (LTextSrcDataSet)null;

                var ctx = System.Web.HttpContext.Current;
                if (ctx != null && ctx.Cache != null)
                {
                    src = ctx.Cache[CACHE_KEY] as LTextSrcDataSet;
                }

                if (src == null)
                {
                    src = new LTextSrcDataSet();
                    System.Web.Caching.CacheDependency cd = null;

                    if (!string.IsNullOrEmpty(LTextMgr.LTextSrcXMLFilePath))
                    {
                        src.LTextSrc.ReadXml(LTextMgr.LTextSrcXMLFilePath);
                        cd = new System.Web.Caching.CacheDependency(LTextMgr.LTextSrcXMLFilePath);
                    }
                    else if (_listener != null)
                    {
                        var arg = new LTextSrcRequestArgs();
                        arg.LTextSrc = src;
                        _listener.RequestLTextSrc(arg);

                        src = arg.LTextSrc;
                        cd = arg.CacheDependency;
                    }

                    src.LTextSrc.DefaultView.Sort = "LangID, LTextKey";

                    if (ctx != null && ctx.Cache != null)
                    {
                        ctx.Cache.Insert(CACHE_KEY, src, cd);
                    }
                }

                return src.LTextSrc;
            }

        }

        /// <summary>
        /// キャッシュをクリアします。データは次回アクセス時に再作成されます。
        /// </summary>
        public static void ClearCache()
        {
            System.Web.HttpContext.Current.Cache.Remove(CACHE_KEY);
        }

        /// <summary>
        /// 現在指定されている言語でローカライズされたテキスト返します。
        /// </summary>
        /// <param name="ltextKey"></param>
        /// <returns></returns>
        public static string GetText(string ltextKey)
        {
            
            return LTextMgr.GetText(Lang.CurrentCulture, ltextKey);
        }

        /// <summary>
        /// 現在指定されている言語でローカライズされたテキスト返します。該当が見つからない場合は defaultText を返します。
        /// </summary>
        /// <param name="ltextKey"></param>
        /// <returns></returns>
        public static string GetText(string ltextKey, string defaultText)
        {
            return LTextMgr.GetText(Lang.CurrentCulture, defaultText);
        }

        /// <summary>
        /// ローカライズされたテキスト返します。該当が見つからない場合は ltextKey を返します。
        /// </summary>
        /// <param name="ltextKey"></param>
        /// <returns></returns>
        public static string GetText(System.Globalization.CultureInfo lang, string ltextKey)
        {
            return LTextMgr.GetText(lang, ltextKey, ltextKey);
        }

        /// <summary>
        /// ローカライズされたテキスト返します。該当が見つからない場合は defaultText を返します。
        /// </summary>
        /// <param name="ltextKey"></param>
        /// <returns></returns>
        public static string GetText(System.Globalization.CultureInfo lang, string ltextKey, string defaultText)
        {
            string text = defaultText;

            DataView dv = LTextSrc.DefaultView;
            var idx = dv.Find(new[] { lang.Name, ltextKey });
            if (idx < 0)
            {
                idx = dv.Find(new[] { lang.TwoLetterISOLanguageName, ltextKey });
            }
            if (idx >= 0)
            {
                text = ((LTextSrcDataSet.LTextSrcRow)dv[idx].Row).Text;
            }

            return text;
        }

        /// <summary>
        /// リスナを登録します。
        /// </summary>
        /// <param name="listener"></param>
        public static void SetListener(ILTextSrcRequestListener listener)
        {
            _listener = listener;
        }
    }
}
